param(
    [string]$ManifestUrl = "https://l2medeina.lt/modpack/manifest.json"
)

Set-StrictMode -Version Latest
$ErrorActionPreference = 'Stop'

trap {
    $errorMessage = "Medeina Launcher failed: $($_.Exception.Message)"
    try {
        Add-Type -AssemblyName PresentationFramework -ErrorAction SilentlyContinue
        [System.Windows.MessageBox]::Show($errorMessage, 'Medeina Launcher Error', 'OK', 'Error') | Out-Null
    }
    catch {
        Write-Host $errorMessage -ForegroundColor Red
        Start-Sleep -Seconds 8
    }

    break
}

$currentApartment = [System.Threading.Thread]::CurrentThread.GetApartmentState().ToString()
if ($currentApartment -ne 'STA') {
    $powershellExe = Join-Path $env:WINDIR 'System32\WindowsPowerShell\v1.0\powershell.exe'
    $argList = @(
        '-NoProfile',
        '-ExecutionPolicy', 'Bypass',
        '-STA',
        '-File', "`"$PSCommandPath`"",
        '-ManifestUrl', "`"$ManifestUrl`""
    )

    Start-Process -FilePath $powershellExe -ArgumentList ($argList -join ' ')
    exit
}

Add-Type -AssemblyName PresentationFramework

$MinecraftDir = Join-Path $env:APPDATA '.minecraft'
$VersionsDir = Join-Path $MinecraftDir 'versions'
$LauncherProfilesPath = Join-Path $MinecraftDir 'launcher_profiles.json'
$OptionsPath = Join-Path $MinecraftDir 'options.txt'

function Set-UiEnabled {
    param(
        [bool]$Enabled,
        [System.Windows.Controls.Button]$CheckButton,
        [System.Windows.Controls.Button]$PlayButton
    )

    $CheckButton.IsEnabled = $Enabled
    $PlayButton.IsEnabled = $Enabled
}

function Get-LocalSha256 {
    param([string]$Path)
    if (-not (Test-Path $Path)) { return $null }
    return (Get-FileHash -Path $Path -Algorithm SHA256).Hash.ToLowerInvariant()
}

function Ensure-TargetDirectory {
    param([string]$FilePath)
    $dir = Split-Path -Parent $FilePath
    if ($dir -and -not (Test-Path $dir)) {
        New-Item -Path $dir -ItemType Directory -Force | Out-Null
    }
}

function Download-File {
    param(
        [string]$Url,
        [string]$Destination
    )

    Ensure-TargetDirectory -FilePath $Destination
    Invoke-WebRequest -Uri $Url -OutFile $Destination -UseBasicParsing
}

function Write-LinesUtf8NoBom {
    param(
        [string]$Path,
        [string[]]$Lines
    )

    $encoding = New-Object System.Text.UTF8Encoding($false)
    [System.IO.File]::WriteAllLines($Path, $Lines, $encoding)
}

function ConvertTo-ManifestObject {
    param($ManifestInput)

    if ($null -eq $ManifestInput) {
        throw 'Manifest response is empty.'
    }

    if ($ManifestInput -is [System.Array]) {
        $ManifestInput = ($ManifestInput -join [Environment]::NewLine)
    }

    if ($ManifestInput -isnot [string] -and ($ManifestInput.PSObject.Properties.Name -contains 'Content')) {
        $ManifestInput = [string]$ManifestInput.Content
    }

    if ($ManifestInput -is [string]) {
        $trimmed = $ManifestInput.Trim().TrimStart([char]0xFEFF)
        if ([string]::IsNullOrWhiteSpace($trimmed)) {
            throw 'Manifest response is empty.'
        }

        $jsonText = $trimmed
        $firstChar = $jsonText.Substring(0, 1)

        if ($firstChar -ne '{' -and $firstChar -ne '[') {
            if ($firstChar -eq '"') {
                try {
                    $jsonText = $jsonText | ConvertFrom-Json
                }
                catch {
                    throw "Manifest is not valid JSON: $($_.Exception.Message)"
                }
            }
            else {
                $snippet = $jsonText.Substring(0, [Math]::Min(80, $jsonText.Length))
                throw "Manifest is not valid JSON: unexpected content '$snippet'"
            }
        }

        try {
            $ManifestInput = $jsonText | ConvertFrom-Json
        }
        catch {
            throw "Manifest is not valid JSON: $($_.Exception.Message)"
        }
    }

    $hasFiles = $ManifestInput.PSObject.Properties.Name -contains 'files'
    $hasBaseUrl = $ManifestInput.PSObject.Properties.Name -contains 'baseUrl'
    $hasLoader = $ManifestInput.PSObject.Properties.Name -contains 'loader'
    if (-not ($hasFiles -and $hasBaseUrl -and $hasLoader)) {
        throw 'Manifest missing required properties (files, baseUrl, loader).'
    }

    return $ManifestInput
}

function Get-Manifest {
    param([string]$Url)

    try {
        $response = Invoke-WebRequest -Uri $Url -Method Get -UseBasicParsing
        return ConvertTo-ManifestObject -ManifestInput $response.Content
    }
    catch {
        try {
            $fallback = Invoke-RestMethod -Uri $Url -Method Get
            return ConvertTo-ManifestObject -ManifestInput $fallback
        }
        catch {
            throw
        }
    }
}

function Get-LauncherProfileNames {
    if (-not (Test-Path $LauncherProfilesPath)) {
        return @()
    }

    try {
        $json = Get-Content -Path $LauncherProfilesPath -Raw | ConvertFrom-Json
        if ($null -eq $json.profiles) { return @() }
        return @($json.profiles.PSObject.Properties.Name)
    } catch {
        return @()
    }
}

function Test-RequiredProfile {
    param([string]$ProfileName)

    if ([string]::IsNullOrWhiteSpace($ProfileName)) { return $false }

    # Strong check: exact version directory exists.
    $versionFolder = Join-Path $VersionsDir $ProfileName
    if (Test-Path $versionFolder) { return $true }

    # Compatibility check: profile id exists in launcher profile set.
    $launcherProfiles = Get-LauncherProfileNames
    return ($launcherProfiles -contains $ProfileName)
}

function Start-MinecraftLauncher {
    param([string]$ProfileName)

    function Get-MinecraftLauncherPath {
        $found = New-Object System.Collections.Generic.List[string]

        $candidates = @(
            "${env:ProgramFiles(x86)}\Minecraft Launcher\MinecraftLauncher.exe",
            "$env:ProgramFiles\Minecraft Launcher\MinecraftLauncher.exe",
            "$env:LOCALAPPDATA\Microsoft\WindowsApps\MinecraftLauncher.exe",
            "$env:LOCALAPPDATA\Microsoft\WindowsApps\minecraftlauncher.exe"
        )

        foreach ($candidate in $candidates) {
            if (-not [string]::IsNullOrWhiteSpace($candidate) -and (Test-Path $candidate) -and (-not $found.Contains($candidate))) {
                [void]$found.Add($candidate)
            }
        }

        try {
            $cmd = Get-Command -Name 'MinecraftLauncher.exe' -ErrorAction SilentlyContinue
            if ($cmd -and (Test-Path $cmd.Source) -and (-not $found.Contains($cmd.Source))) {
                [void]$found.Add($cmd.Source)
            }
        } catch { }

        $uninstallRoots = @(
            'HKCU:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*',
            'HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*',
            'HKLM:\Software\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*'
        )

        foreach ($root in $uninstallRoots) {
            try {
                $entries = @(Get-ItemProperty -Path $root -ErrorAction SilentlyContinue | Where-Object {
                    [string]$_.DisplayName -like '*Minecraft Launcher*'
                })

                foreach ($entry in $entries) {
                    $displayIcon = [string]$entry.DisplayIcon
                    if (-not [string]::IsNullOrWhiteSpace($displayIcon)) {
                        $iconPath = ($displayIcon.Split(',')[0]).Trim().Trim('"')
                        if ((Test-Path $iconPath) -and (-not $found.Contains($iconPath))) {
                            [void]$found.Add($iconPath)
                        }
                    }

                    $installLocation = [string]$entry.InstallLocation
                    if (-not [string]::IsNullOrWhiteSpace($installLocation)) {
                        foreach ($exeName in @('MinecraftLauncher.exe', 'Minecraft.exe')) {
                            $path = Join-Path $installLocation $exeName
                            if ((Test-Path $path) -and (-not $found.Contains($path))) {
                                [void]$found.Add($path)
                            }
                        }
                    }
                }
            } catch { }
        }

        return $found | Select-Object -First 1
    }

    $launcherPath = Get-MinecraftLauncherPath
    if ($launcherPath) {
        # Try launching directly into the profile; if not supported by launcher build,
        # it still opens launcher and user can press Play.
        Start-Process -FilePath $launcherPath -ArgumentList "--launchProfile `"$ProfileName`""
        return
    }

    # Fallback for Microsoft Store installations where executable path is virtualized.
    try {
        Start-Process -FilePath 'explorer.exe' -ArgumentList 'shell:AppsFolder\Microsoft.4297127D64EC6_8wekyb3d8bbwe!Minecraft'
        return
    } catch { }

    # Last resort: URI protocol handler.
    try {
        Start-Process -FilePath 'minecraft://'
        return
    } catch { }

    throw 'Minecraft Launcher not found. Install the official launcher or enable the MinecraftLauncher.exe App Execution Alias in Windows Settings.'
}

function Get-ManifestResourcePackEntries {
    param($Manifest)

    $entries = New-Object System.Collections.Generic.List[string]

    foreach ($f in @($Manifest.files)) {
        $path = [string]$f.path
        if ($path -match '^resourcepacks/(.+)$') {
            $resourcePackPath = ($Matches[1] -replace '\\', '/')
            if (-not [string]::IsNullOrWhiteSpace($resourcePackPath)) {
                $entry = "file/$resourcePackPath"
                if (-not $entries.Contains($entry)) {
                    [void]$entries.Add($entry)
                }
            }
        }
    }

    return @($entries)
}

function Ensure-ResourcePackOptions {
    param(
        $Manifest,
        [System.Windows.Controls.TextBlock]$StatusText
    )

    $manifestPacks = @(Get-ManifestResourcePackEntries -Manifest $Manifest)
    if ($manifestPacks.Count -eq 0) {
        return
    }

    Ensure-TargetDirectory -FilePath $OptionsPath

    $lines = @()
    if (Test-Path $OptionsPath) {
        $lines = @(Get-Content -Path $OptionsPath)
    }

    $lineMap = @{}
    for ($i = 0; $i -lt $lines.Count; $i++) {
        $line = $lines[$i]
        $sepIndex = $line.IndexOf(':')
        if ($sepIndex -gt 0) {
            $key = $line.Substring(0, $sepIndex)
            if (-not $lineMap.ContainsKey($key)) {
                $lineMap[$key] = $i
            }
        }
    }

    $currentResourcePacks = @('vanilla')
    if ($lineMap.ContainsKey('resourcePacks')) {
        $rawResourcePacks = $lines[[int]$lineMap['resourcePacks']].Substring('resourcePacks:'.Length)
        try {
            $parsed = @($rawResourcePacks | ConvertFrom-Json)
            if ($parsed.Count -gt 0) {
                $currentResourcePacks = @($parsed)
            }
        }
        catch {
            $currentResourcePacks = @('vanilla')
        }
    }

    foreach ($pack in $manifestPacks) {
        if ($currentResourcePacks -notcontains $pack) {
            $currentResourcePacks += $pack
        }
    }

    $resourcePacksJson = $currentResourcePacks | ConvertTo-Json -Compress
    $resourcePacksLine = "resourcePacks:$resourcePacksJson"

    if ($lineMap.ContainsKey('resourcePacks')) {
        $lines[[int]$lineMap['resourcePacks']] = $resourcePacksLine
    }
    else {
        $lines += $resourcePacksLine
    }

    $currentIncompatible = @()
    if ($lineMap.ContainsKey('incompatibleResourcePacks')) {
        $rawIncompatible = $lines[[int]$lineMap['incompatibleResourcePacks']].Substring('incompatibleResourcePacks:'.Length)
        try {
            $currentIncompatible = @($rawIncompatible | ConvertFrom-Json)
        }
        catch {
            $currentIncompatible = @()
        }
    }

    $filteredIncompatible = @($currentIncompatible | Where-Object { $manifestPacks -notcontains [string]$_ })
    $incompatibleJson = $filteredIncompatible | ConvertTo-Json -Compress
    $incompatibleLine = "incompatibleResourcePacks:$incompatibleJson"

    if ($lineMap.ContainsKey('incompatibleResourcePacks')) {
        $lines[[int]$lineMap['incompatibleResourcePacks']] = $incompatibleLine
    }
    else {
        $lines += $incompatibleLine
    }

    Write-LinesUtf8NoBom -Path $OptionsPath -Lines $lines
    $StatusText.Text = 'Updated options.txt to auto-enable resource pack(s).'
}

function Sync-Pack {
    param(
        $Manifest,
        [System.Windows.Controls.TextBlock]$StatusText,
        [System.Windows.Controls.ProgressBar]$ProgressBar
    )

    $files = @($Manifest.files)
    $total = [Math]::Max($files.Count, 1)
    $index = 0

    foreach ($obsolete in @($Manifest.remove)) {
        $target = Join-Path $MinecraftDir ($obsolete -replace '/', '\\')
        if (Test-Path $target) {
            Remove-Item -Path $target -Force
        }
    }

    foreach ($f in $files) {
        $index++
        $relativePath = ($f.path -replace '/', '\\')
        $target = Join-Path $MinecraftDir $relativePath
        $remoteUrl = $Manifest.baseUrl.TrimEnd('/') + '/' + $f.path

        $StatusText.Text = "Checking $($f.path)..."
        $ProgressBar.Value = [int](($index - 1) * 100 / $total)

        $localHash = Get-LocalSha256 -Path $target
        $expected = ([string]$f.sha256).ToLowerInvariant()

        if ($localHash -ne $expected) {
            $StatusText.Text = "Downloading $($f.path)..."
            Download-File -Url $remoteUrl -Destination $target

            $newHash = Get-LocalSha256 -Path $target
            if ($newHash -ne $expected) {
                throw "Hash validation failed: $($f.path)"
            }
        }
    }

    $ProgressBar.Value = 100
}

[xml]$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        Title="Medeina Mod Launcher" Height="260" Width="460"
        WindowStartupLocation="CenterScreen" ResizeMode="NoResize">
  <Grid Margin="16">
    <Grid.RowDefinitions>
      <RowDefinition Height="Auto"/>
      <RowDefinition Height="Auto"/>
      <RowDefinition Height="Auto"/>
      <RowDefinition Height="*"/>
    </Grid.RowDefinitions>

    <TextBlock Grid.Row="0" FontSize="18" FontWeight="Bold" Text="Medeina Mod Launcher"/>
    <TextBlock Grid.Row="1" Margin="0,8,0,8" Name="StatusText" Text="Ready." TextWrapping="Wrap"/>
    <ProgressBar Grid.Row="2" Name="ProgressBar" Height="20" Minimum="0" Maximum="100" Value="0"/>

    <StackPanel Grid.Row="3" Orientation="Horizontal" HorizontalAlignment="Right" Margin="0,16,0,0">
      <Button Name="CheckButton" Width="130" Height="34" Margin="0,0,8,0">Check Updates</Button>
      <Button Name="PlayButton" Width="130" Height="34">Play</Button>
    </StackPanel>
  </Grid>
</Window>
"@

$reader = New-Object System.Xml.XmlNodeReader $xaml
$window = [Windows.Markup.XamlReader]::Load($reader)

$statusText = $window.FindName('StatusText')
$progressBar = $window.FindName('ProgressBar')
$checkButton = $window.FindName('CheckButton')
$playButton = $window.FindName('PlayButton')

$script:ManifestCache = $null

$checkButton.Add_Click({
    try {
        Set-UiEnabled -Enabled $false -CheckButton $checkButton -PlayButton $playButton
        $statusText.Text = 'Fetching manifest...'
        $progressBar.Value = 5

        $manifest = Get-Manifest -Url $ManifestUrl
        $script:ManifestCache = $manifest

        Sync-Pack -Manifest $manifest -StatusText $statusText -ProgressBar $progressBar
        Ensure-ResourcePackOptions -Manifest $manifest -StatusText $statusText

        if (-not (Test-RequiredProfile -ProfileName $manifest.loader.profileName)) {
            throw "Required profile '$($manifest.loader.profileName)' is missing. Install the correct loader first."
        }

        $statusText.Text = "Up to date (v$($manifest.packVersion)). Ready to play."
    }
    catch {
        $statusText.Text = "Update failed: $($_.Exception.Message)"
    }
    finally {
        Set-UiEnabled -Enabled $true -CheckButton $checkButton -PlayButton $playButton
    }
})

$playButton.Add_Click({
    try {
        Set-UiEnabled -Enabled $false -CheckButton $checkButton -PlayButton $playButton

        if (-not $script:ManifestCache) {
            $statusText.Text = 'Fetching manifest...'
            $script:ManifestCache = Get-Manifest -Url $ManifestUrl
        }

        $required = [string]$script:ManifestCache.loader.profileName
        if (-not (Test-RequiredProfile -ProfileName $required)) {
            throw "Required profile '$required' is missing. Install loader before launch."
        }

        Ensure-ResourcePackOptions -Manifest $script:ManifestCache -StatusText $statusText

        $statusText.Text = "Launching Minecraft profile '$required'..."
        Start-MinecraftLauncher -ProfileName $required
    }
    catch {
        $statusText.Text = "Launch failed: $($_.Exception.Message)"
    }
    finally {
        Set-UiEnabled -Enabled $true -CheckButton $checkButton -PlayButton $playButton
    }
})

$window.ShowDialog() | Out-Null
